/* ***** BEGIN LICENSE BLOCK ***** 
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the pRDFData.
 *
 * The Initial Developer of the Original Code is SHIMODA Hiroshi.
 * Portions created by the Initial Developer are Copyright (C) 2002-2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): SHIMODA Hiroshi <piro@p.club.ne.jp>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */


/*
++Abstract:

a class for operate RDF datasource with an RDF container.
the constructor needs 7 parameters when create instance.
	* the ID of the container
	* the URI of the RDF datasource
	* type of container (seq|bag|alt, default is seq)
	* namespace URI (optional)
	* prefix URI of IDs of items (optional)
	* format of ID of data (optional, see the instruction following)
	* a boolean value, indicates the wrapper should flush RDF datasource
	  when you edit data
	  (optional, default is false, see the instruction following)


++Example:

var obj = new pRDFData(
		'DATA',
		'file:///e:/extensions.rdf'
	);

obj.setData('item1', 'Value', 'foobar');
obj.setData('item2', 'Value', 'test');

var value  = obj.getData('item1', 'Value'), // you can get "foobar"
	length = obj.length; // you can get '2'

for (var i = 0; i < obj.length; i++)
{
	alert(obj.getData(obj.item(i), 'Value')
}


++Format of RDF files generated by Mozilla:

After these operations above, the resource was formatted like
following:
<RDF:Seq about="urn:DATA:root"
	xmlns:DATANS="http://white.sakura.ne.jp/~piro/rdf#">
	<RDF:li>
		<RDF:Description about="urn:DATA:item1"
			DATANS:Value="foobar"/>
	</RDF:li>
	<RDF:li>
		<RDF:Description about="urn:DATA:item2"
			DATANS:Value="test"/>
	</RDF:li>
</RDF:Seq>


++Appendix:
nsIRDF* objects have some problems, so you should not use instances of
this class. For example, if you create an instance with "ONLOAD",
don't use it in the same "ONLOAD", so you should use it with a delayed
function, like "setTimeout(func, 0)".


IDs of data is specified like:
'%namespaceURI%urn:%containerID%:%itemID%' (default)

%namespaceURI% : namespace URI of the items
%containerID%  : ID of the container (like "DATA")
%itemID%       : ID of the item

%itemID% is required. If there is no the parameter, it is appended at
the last of item ID automatically.


The ID of an item you specified is stored as a encoded text as URI.
If you want to use ID with other format, like bookmarks, please get
the resource getResource(aName) and operate with getData, setData.


If you give "true" as the seventh argument, this wrapper updates the
datasource file when you edit data by some methods. (setData, etc.)
If not, or "false" is given, the datasource will be updated when
the window is closed. To update the file on demand, use the "flush()"
method with an argument, "true".

*/
/*


TvF
RDFReiƃZbgŃf[^\[X𑀍삷NXB
CX^XɈȉ̃p[^vB
	EReiID
	Ef[^\[XURI
	EReĩ^Cviseq|bag|alt, ȗ, ftHgseqj
	EOԂURIiȗj
	E\[X̃x[XURIiȗj
	Eۑ郊\[Xiď`iȗFڂ͉QƁj
	Ef[^ҏWƂɃt@CXV邩ǂ^Ul
	  iȗ, ftHgfalse, ڂ͉LQƁj


gpF
var obj = new pRDFData(
		'DATA',
		'file:///e:/extensions.rdf'
	);

obj.setData('item1', 'Value', 'قق');
obj.setData('item2', 'Value', '͂͂');

var value  = obj.getData('item1', 'Value'), // 'قق' 
	length = obj.length; // '2'

// o^ς݂̑S\[XɂĒl𒲂ׂ
for (var i = 0; i < obj.length; i++)
{
	alert(obj.getData(obj.item(i), 'Value')
}


RDFt@Č`F
L̑siKŁAȊRDF\[XłĂƎv
B
<RDF:Seq about="urn:DATA:root"
	xmlns:DATANS="http://white.sakura.ne.jp/~piro/rdf#">
	<RDF:li>
		<RDF:Description about="urn:DATA:item1"
			DATANS:Value="قق"/>
	</RDF:li>
	<RDF:li>
		<RDF:Description about="urn:DATA:item2"
			DATANS:Value="͂͂"/>
	</RDF:li>
</RDF:Seq>


ߑF
̏̊֌WAĂяoĂɂ͎gȂłB
insIRDF*̊eIuWFNg̃oOidlHj΍j
onloadȂǂŃCX^X𐶐ꍇAonload̏ł͌ĂяoA
setTimeout(func, 0) Ȃǂŏx点ƁA삵܂B


ۑe\[Xiď`́Â悤ȏŕŎw肵܂B
'%namespaceURI%urn:%containerID%:%itemID%' iftHg͂̌`j

%namespaceURI% : OURI
%containerID%  : ReiIDił"DATA"j
%itemID%       : ACeID

%itemID% ͕K{ڂłBw̒ɂꂪ܂܂ĂȂꍇ́A
ɏɉ܂BȂAReiID͕̏ύXł܂B


ACeIDURIpɃGR[hĕۑ܂BGR[hĂ͍
ꍇiubN}[NURIƓIDgƂȂǁj́AgetResource(aName)Ń
\[X擾ĂID̑̕getData, setDataȂǂɓnĉ
BȂAۑς݃ACeɂĂitem(n)ł\[X擾ł܂B


7̈ƂtruenƁAsetDataȂǂŃf[^ҏWۂɑ
RDFf[^\[X̃t@CōXV܂Bw肠邢falseƁA
EBhE܂ŁA邢flush()\bhtrueƂēn
g܂ŁAt@C͍XV܂B

*/


function pRDFData(aID, aDataSourceURI, aType, aNS, aRsourceBaseURI, aIDFormat, aShouldFlushOnEdit)
{
	this.id          = aID; // urn:<id>:root A<id>̕ɂȂ
	this.type        = (aType) ? aType.toLowerCase() : 'seq' ;
	this.NS          = aNS || 'http://white.sakura.ne.jp/~piro/rdf#';
	this.resURIPrefix = aRsourceBaseURI || '' ;
	this.id_format   = aIDFormat || '%namespaceURI%urn:%containerID%:%itemID%' ;
	this.shouldFlush = aShouldFlushOnEdit || false ;

	if (!this.id_format.match(/%itemID%/)) this.id_format += '%itemID%';

	var ProfD = this.getURISpecFromKey('ProfD');
	if (!ProfD.match(/\/$/)) ProfD += '/';
	this.dsource_uri = (aDataSourceURI && aDataSourceURI.match(/^\w+:/)) ? aDataSourceURI : ProfD+aDataSourceURI ;

	this._resources = [];

	this.init();
}

pRDFData.dataSourcesShouldBeFlushed = [];
pRDFData.onUnloadRegistered = false;
pRDFData.onUnload           = function()
{
	for (var i in pRDFData.dataSourcesShouldBeFlushed)
		pRDFData.dataSourcesShouldBeFlushed[i].flush(true);
};


pRDFData.prototype =
{
	shouldFlushOnUnload : false,

	RDF       : Components.classes['@mozilla.org/rdf/rdf-service;1'].getService(Components.interfaces.nsIRDFService),
	RDFCUtils : Components.classes['@mozilla.org/rdf/container-utils;1'].getService(Components.interfaces.nsIRDFContainerUtils),

	container : null,
	
	// Initializing 
	// 
	init : function()
	{
		this.resURI = this.id_format.replace(/%namespaceURI%/gi, this.resURIPrefix).replace(/%containerID%/gi, this.id);


		this.dsource = this.RDF.GetDataSource(this.dsource_uri);

		this.container = Components.classes['@mozilla.org/rdf/container;1'].createInstance(Components.interfaces.nsIRDFContainer);

		this.makeContainer();
		this.reset();

		window.setTimeout(this.initWithDelay, 0, this);
	},

	initWithDelay : function(aObject)
	{
		aObject.reset();
		if (!aObject.length) {
			// remove empty container, because we cannot append element to the empty container which is *loaded*. So, delete the container and re-create with startup.
			aObject.removeResource(aObject.containerNode);
			aObject.makeContainer();
			aObject.reset();
		}
	},
	
	makeContainer : function() 
	{
		this.containerNode = this.RDF.GetResource(this.resURIPrefix+'urn:'+this.id+':root');
		switch(this.type)
		{
			default:
			case 'seq':
				this.RDFCUtils.MakeSeq(this.dsource, this.containerNode);
				break;
			case 'bag':
				this.RDFCUtils.MakeBag(this.dsource, this.containerNode);
				break;
			case 'alt':
				this.RDFCUtils.MakeAlt(this.dsource, this.containerNode);
				break;
		}
		this.container.Init(this.dsource, this.containerNode);
	},
 
	// reset container and array 
	// RDFReiƔz
	reset : function()
	{
		try {
			this.container.Init(this.dsource, this.containerNode);
		}
		catch(e) {
			return false;
		}

		var nodes = this.container.GetElements(),
			res;
		this._resources = [];
		while (nodes.hasMoreElements())
		{
			res = this.RDF.GetResource(nodes.getNext().QueryInterface(Components.interfaces.nsIRDFResource).Value);
			this._resources.push(res);
		}

		return true;
	},
  
	// basic operations 
	// \[X̊{
	
	// get a resource from id 
	// w肵Õ\[X𓾂
	getResource : function(aIDOrResource)
	{
		try { // if a resource is handed, return it
			if (aIDOrResource && 'QueryInterface' in aIDOrResource) {
				aIDOrResource = aIDOrResource.QueryInterface(Components.interfaces.nsIRDFResource);
				return aIDOrResource
			}
		}
		catch(e) {
		}

		return this.RDF.GetResource(this.resURI.replace(/%itemID%/gi, escape(aIDOrResource)));
	},
 
	// get a resource from index 
	// nڂ̃\[X𓾂
	item : function(aIndex)
	{
		return (aIndex >= this.length) ? null : this._resources[aIndex] ;
	},
 
	// get an id from resource 
	// w肵\[Xid𓾂
	getID : function(aResource)
	{
		return unescape(aResource.Value.replace(/^([^#]+#)?urn:[^:]*:/, ''));
	},
 
	// get an index from resource 
	// \[XRDFRei̒ɂ΁ÄʒuԂ
	indexOf : function(aResourceOrID)
	{
		var index = this.container.IndexOf(this.getResource(aResourceOrID));

		return (index > 0) ? index-1 : index ;
	},
  
	// save data 
	// f[^̕ۑ
	
	// save data to the resource 
	// if the resource is only specified, remove the resource.
	// ex. data.setData(aIDorResource, aKey1, aValue1, aKey2, aValue2, ...)
	// w肳ꂽ\[XɁAf[^ۑBlȂꍇAf[^폜B
	// \[XBćAL[ƒl̑΁B\[X͖OŎw肷邱Ƃ\B
	// \[XnꍇA\[X̂폜B
	setData : function()
	{
		var arg          = (arguments[0].constructor == Array) ? arguments[0] : this.concat(arguments);
		var resourceOrID = arg.shift();
		var removedData  = [];

		if (!resourceOrID)
			return removedData;

		resourceOrID = this.getResource(resourceOrID);

		var valueName,
			newValue,
			oldValue;

		switch (arg.length)
		{
			case 0:
				try {
					this.container.RemoveElement(resourceOrID, true);
					removedData = this.removeResource(resourceOrID);
				}
				catch(e) {
//					dump(e+'\n');
				}
				break;

			default:
				for (var i = 0; i < arg.length; i += 2)
				{
					valueName = this.RDF.GetResource(this.NS+arg[i]);
					try { // Âf[^폜
						oldValue = this.dsource.GetTarget(resourceOrID, valueName, true);

						removedData[arg[i]] = oldValue.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;

						this.dsource.Unassert(resourceOrID, valueName, oldValue);
					}
					catch(e) {
//						dump(e+'\n');
					}

					// lundefined̏ꍇA폜邾ŏI
					if (arg[i+1] === void(0)) continue;

					newValue  = this.RDF.GetLiteral(arg[i+1] === null ? '' : String(arg[i+1]));
					this.dsource.Assert(resourceOrID, valueName, newValue, true);
				}
				if (this.indexOf(resourceOrID) < 0)
					this.container.AppendElement(resourceOrID);

				break;
		}
		this.reset();
		this.flush();

		return removedData;
	},
 
	// save file to local disk 
	// t@C̍XV
	flush : function(aForce)
	{
		if (!aForce && !this.shouldFlush) { // flush on unload
			if (this.shouldFlushOnUnload) return;

			pRDFData.dataSourcesShouldBeFlushed[this.dsource_uri] = this;

			if (!pRDFData.onUnloadRegistered)
				window.addEventListener(
					'unload',
					pRDFData.onUnload,
					false
				);

			this.shouldFlushOnUnload = true;
		}
		else
			this.dsource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
	},
  
	// load data 
	// \[Xf[^𓾂
	
	// get stored data of a resource with a key 
	// w肳ꂽ\[XAL[̒l𓾂B
	// \[X͖OŎw肷邱Ƃ\B
	getData : function(aResourceOrID, valueKey)
	{
		aResourceOrID = this.getResource(aResourceOrID);

		var value;
		try {
			value = this.dsource.GetTarget(
						aResourceOrID,
						this.RDF.GetResource(this.NS+valueKey),
						true
					);
			value = (value) ? value.QueryInterface(Components.interfaces.nsIRDFLiteral).Value : '' ;
		}
		catch(e) {
			value = '';
		}
		return value;
	},
 
	// get stored data of a resource of directory with a key 
	// ^ꂽURLwփfBNgHĂAł߂fBNg̃f[^Ԃ
	getDataFromPath : function(aPath, aKey)
	{
		var ret = '';
		if (!aPath) return '';

		if (aPath.match(/^[^:\/]+:[^:\/]+:\//i) ||
			aPath.match(/^(view-source|about|jar):/i)) return '';

		var dir = this.getParentDirs(aPath);
		for (var i in dir)
		{
			ret = this.getData(dir[i], aKey);
			if (ret) break;
		}
		return ret;
	},
  
	// operate data 
	// \[X̑
	
	// delete data 
	// we can specfy the data by ID
	// \[X̍폜
	// \[X͖OŎw肷邱Ƃ\B
	removeData : function(aResourceOrID)
	{
		aResourceOrID = this.getResource(aResourceOrID);

		return this.setData(aResourceOrID);
	},
	
	// delete resource 
	removeResource : function(aResourceOrID)
	{
		aResourceOrID = this.getResource(aResourceOrID);

		var removedData = [];
		var names = this.dsource.ArcLabelsOut(aResourceOrID),
			name,
			value,
			removed;
		// UnassertőSẴf[^폜ƁARDF:DescriptionIɍ폜B
		while (names.hasMoreElements())
		{
			try {
				name = names.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
				value = this.dsource.GetTarget(aResourceOrID, name, true);

				// if the removed data is a literal string, add to the log
				try {
					removed = value.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
					removedData[name.Value.split('#')[1]] = removed;
				}
				catch(e) {
				}

//				dump('REMOVED: '+this.id+' / '+name.Value+'\n');

				this.dsource.Unassert(aResourceOrID, name, value);
			}
			catch(e) {
//				dump('FAIL TO REMOVE: '+this.id+'\n');
//				dump(e+'\n');
			}
		}

		return removedData;
	},
  
	// copy a data to new data 
	// \[XRs[
	copyData : function(aOldResourceOrName, aNewaName)
	{
		if (!aOldResourceOrName || !aNewaName) return;

		aOldResourceOrName = this.getResource(aOldResourceOrName);

		var names = this.dsource.ArcLabelsOut(aOldResourceOrName),
			name;
		while (names.hasMoreElements())
		{
			name = names.getNext().QueryInterface(Components.interfaces.nsIRDFResource).Value.match(/[^#]+$/);
			this.setData(aNewaName, name, this.getData(aOldResourceOrName, name));
		}
		return;
	},
 
	// rename a data 
	// \[X̃l[
	renameData : function(aOldResourceOrName, aNewaName)
	{
		if (!aOldResourceOrName || !aNewaName ||
			this.getData(aNewaName, 'Name')) // if the item exists, do no action
			return null;

		var position = this.indexOf(aOldResourceOrName);
		this.copyData(aOldResourceOrName, aNewaName);
		this.removeData(aOldResourceOrName);
		this.moveElementTo(aNewaName, position);
		this.setData(aNewaName, 'Name', aNewaName);

		return aOldResourceOrName;
	},
 
	// move a data to (with absolute index) 
	// \[X̓o^<index>ɕύXij
	moveElementTo : function(aResourceOrID, aIndex)
	{
		aResourceOrID = this.getResource(aResourceOrID);

		var current = this.indexOf(aResourceOrID);

		if (current < 0 ||
			aIndex == current ||
			aIndex < 0 ||
			aIndex >= this.length) return false;

		this.container.RemoveElementAt(current+1, true);
		this.container.InsertElementAt(aResourceOrID, aIndex+1, true);

		this.reset();
		this.flush();
		return true;
	},
 
	// move a data by (relative index) 
	// \[X̓o^<order>
	// \[X͖OŎw肷邱Ƃ\B
	// ȂꍇAfalseԂ
	moveElement : function(aResourceOrID, aOrder)
	{
		aResourceOrID = this.getResource(aResourceOrID);

		var index   = this.indexOf(aResourceOrID),
			toIndex = index+aOrder;

		if (index < 0) return false;

		return this.moveElementTo(aResourceOrID, toIndex);
	},
 
	// clear all data 
	// SẴ\[X̍폜
	clearData : function()
	{
		var count = this.length;
		if (count)
			for (var i = count-1; i > -1; i--)
				this.removeData(this.item(i));
	},
 
	// clean up the datasource (delete all of garbages) 
	// Ou:rootvŏIĂ炸eĂȂǗ\[XSď
	cleanUp : function(aForce)
	{
		var resources = this.dsource.GetAllResources();
		var resource;
		while (resources.hasMoreElements())
		{
			resource = resources.getNext().QueryInterface(Components.interfaces.nsIRDFResource);

			if (!resource.Value.match(/:root$/) &&
				!this.dsource.ArcLabelsIn(resource).hasMoreElements())
				this.removeResource(resource);
		}

		this.flush(aForce);
	},
  
	// common methods 
	// ėp\bhEvpeB
	
	getURISpecFromKey : function(aKey) 
	{
		const DIR = Components.classes['@mozilla.org/file/directory_service;1'].getService(Components.interfaces.nsIProperties);
		var dir = DIR.get(aKey, Components.interfaces.nsILocalFile);
		return this.getURLSpecFromFilePath(dir.path);
	},
	
	getURLSpecFromFilePath : function(aPath) 
	{
		var tempLocalFile = this.makeFileWithPath(aPath);

		try {
			return this.IOService.newFileURI(tempLocalFile).spec;
		}
		catch(e) { // [[interchangeability for Mozilla 1.1]]
			return this.IOService.getURLSpecFromFile(tempLocalFile);
		}
	},
  
	makeFileWithPath : function(aPath) 
	{
		var newFile = Components.classes['@mozilla.org/file/local;1'].createInstance(Components.interfaces.nsILocalFile);
		newFile.initWithPath(aPath);
		return newFile;
	},
 
	createSupportsArray : function() 
	{
		return Components.classes['@mozilla.org/supports-array;1'].createInstance(Components.interfaces.nsISupportsArray);
	},
 
	concat : function() 
	{
		var ret = [],
			i, j;
		for (i = 0; i < arguments.length; i++)
			for (j = 0; j < arguments[i].length; j++)
				ret.push(arguments[i][j]);

		return ret;
	},
 
	// extract the current directory from an URI 
	// URIfBNg𔲂o
	getCurrentDir : function(aURI)
	{
		return (aURI) ? aURI.replace(/[#?].*|[^\/]*$/g, '') : '' ;
	},
	
	// extract the parent directory from an URI 
	// URIefBNg𔲂o
	getParentDir : function(aURI)
	{
		var dir = this.getCurrentDir(aURI || '');
		return (aURI && dir == aURI.match(/^[^:]+:\/\/[^\/]*\//)) ? dir : dir.replace(/[^\/]*\/$/, '') ;
	},
	
	// extract parent directories from an URI and return an array 
	// efBNg[g܂ŒHAzƂĕԂ
	getParentDirs : function(aURI)
	{
		var dir = [this.getCurrentDir(aURI || '')];
		for (var i = 1; dir[i-1] != this.getParentDir(dir[i-1]); i++)
			dir[i] = this.getParentDir(dir[i-1]);
		return dir;
	},
   
	get IOService() 
	{
		if (!this._IOService) {
			this._IOService = Components.classes['@mozilla.org/network/io-service;1'].getService(Components.interfaces.nsIIOService);
		}
		return this._IOService;
	},
	_IOService : null,
  
	// methods compatible Array 
	// Array IvpeBE\bh
	
	// length 
	get length()
	{
		try {
			return this.container.GetCount();
		}
		catch(e) {
			return 0;
		}
	},
 
	// push(); 
	push : function()
	{
		var resources = this.concat(arguments);
		for (var i in resources)
			this.container.AppendElement(resources[i]);

		this.reset();
		return this.length;
	},
 
	// unshift(); 
	unshift : function() {
		var resources = this.concat(arguments);
		for (var i in resources)
			this.container.InsertElementAt(resources[i], i, true);

		this.reset();
		return this.length;
	},
 
	// pop(); 
	pop : function()
	{
		var resource = this.item(obj.length-1);
		this.container.RemoveElement(resource, true);
		this.reset();
		return resource;
	},
 
	// shift(); 
	shift : function() {
		var resource = this.item(0);
		this.container.RemoveElement(resource, true);
		this.reset();
		return resource;
	},
  
	// methods compatible DOM 
	// DOM-Node IvpeBE\bh
	
	// appendChild(); 
	appendChild : function(aResource)
	{
		this.container.AppendElement(aResource);
		this.reset();
		return aResource;
	},
 
	// insertBefore(); 
	insertBefore : function(aResource, aRefResource)
	{
		this.container.InsertElementAt(aResource, this.indexOf(aRefResource), true);
		this.reset();
		return aResource;
	}
   
// 
};
 
